// init name space
var pbs = pbs ? pbs : {};

// define all prototypes within an anonymous self executing fuction
(function(pbs, undefined){
	//
	// ==== Define Needed Helper Functions ===
	//
	
	// A function for validating integer inputs
	function isValidInteger(v, lbound, ubound){
		// first and foremost, make sure we have an integer
		if(!String(v).match(/^-?\d+$/)){
			return false;
		}
		
		// if a lower bound was passed, check it
		if(typeof lbound === 'number' && v < lbound){
			return false;
		}
		
		// if an upper bound was passed, check it
		if(typeof ubound === 'number' && v > ubound){
			return false;
		}
		
		// if we got here all is well
		return true;
	}
	
	// a data structure to help validate days of the month
	var daysInMonthLookup = {};
	daysInMonthLookup[1] = 31;
	daysInMonthLookup[2] = 28;
	daysInMonthLookup[3] = 31;
	daysInMonthLookup[4] = 30;
	daysInMonthLookup[5] = 31;
	daysInMonthLookup[6] = 30;
	daysInMonthLookup[7] = 31;
	daysInMonthLookup[8] = 31;
	daysInMonthLookup[9] = 30;
	daysInMonthLookup[10] = 31;
	daysInMonthLookup[11] = 30;
	daysInMonthLookup[12] = 31;
	
	// helper function to validate a given combination of day, month, and year
	function isValidateDMYCombo(d, m, y){
		// figure out how many days are allowed in the curreny month
		var numDaysInMonth = daysInMonthLookup[m];
		if(m === 2){
			// the month is February, so check for a leap year and
			// if we are a leap year, change the days to 29
			if(pbs.Date.isLeapYear(y)){
				numDaysInMonth = 29;
			}
		}
		
		// return based on wheather or not the days are valid
		return d <= numDaysInMonth ? true : false;
	}
	
	// helper function to convert integers to zero-padded strings
	function intToPaddedString(i, len){
		// take note of whethere or not the original number was negative
		var isNegative = i < 0 ? true : false;
		
		// convert the absolute value of the number to a string
		var ans = String(Math.abs(i));
		
		// add any needed padding if a sane length was provided
		if(typeof len === 'number' && len > 0){
			while(ans.length < len){
				ans = '0' + ans;
			}
		}
		
		// pre-fix the minus sign if needed
		if(isNegative){
			ans = '-' + ans;
		}
		
		return ans;
	}
	
	// a helper function to get the two-letter ordinal suffix for any integer
	function toOrdinalString(n){
		if(n === 1){
			return 'st';
		}
		if(n === 2){
			return 'nd';
		}
		if(n === 3){
			return 'rd';
		}
		return 'th';
	}
	
	// a lookup table to convert month numbers into English names
	var monthNameLookup = {};
	monthNameLookup[1] = 'January';
	monthNameLookup[2] = 'February';
	monthNameLookup[3] = 'March';
	monthNameLookup[4] = 'April';
	monthNameLookup[5] = 'May';
	monthNameLookup[6] = 'June';
	monthNameLookup[7] = 'July';
	monthNameLookup[8] = 'August';
	monthNameLookup[9] = 'September';
	monthNameLookup[10] = 'October';
	monthNameLookup[11] = 'November';
	monthNameLookup[12] = 'December';
	
	//
	// === Define the Time Prototype ===
	//
	
	// the constructor
	pbs.Time = function(h, m, s){
		// init data with default values
		this._hours = 0;
		this._minutes = 0;
		this._seconds = 0;
		
		// process any args that were passed
		if(typeof h !== 'undefined'){
			this.hours(h);
		}
		if(typeof m !== 'undefined'){
			this.minutes(m);
		}
		if(typeof s !== 'undefined'){
			this.seconds(s);
		}
	};
	
	// the accessor methods
	pbs.Time.prototype.hours = function(h){
		if(arguments.length === 0){
			return this._hours;
		}
		if(!isValidInteger(h, 0, 23)){
			throw new TypeError('the hours value must be an integer between 0 and 23 inclusive');
		}
		this._hours = h;
		return this;
	};
	pbs.Time.prototype.minutes = function(m){
		if(arguments.length === 0){
			return this._minutes;
		}
		if(!isValidInteger(m, 0, 59)){
			throw new TypeError('the minutes value must be an integer between 0 and 59 inclusive');
		}
		this._minutes = m;
		return this;
	};
	pbs.Time.prototype.seconds = function(s){
		if(arguments.length === 0){
			return this._seconds;
		}
		if(!isValidInteger(s, 0, 59)){
			throw new TypeError('the seconds value must be an integer between 0 and 59 inclusive');
		}
		this._seconds = s;
		return this;
	};
	
	// add functions
	pbs.Time.prototype.time12 = function(){
		var ans = '';
		if(this._hours === 0){
			ans += '12';
		}else if(this._hours <= 12){
			ans += this._hours;
		}else{
			ans += (this._hours - 12);
		}
		ans += ':' + intToPaddedString(this._minutes, 2) + ':' + intToPaddedString(this._seconds, 2);
		ans += this._hours < 12 ? 'AM' : 'PM';
		return ans;
	};
	pbs.Time.prototype.time24 = function(){
		return '' + intToPaddedString(this._hours, 2) + ':' + intToPaddedString(this._minutes, 2) + ':' + intToPaddedString(this._seconds, 2);
	};
	
	// define a toString function
	pbs.Time.prototype.toString = pbs.Time.prototype.time24;
	
	// define a clone function
	pbs.Time.prototype.clone = function(){
		return new pbs.Time(this._hours, this._minutes, this._seconds);
	};
	
	// define comparison functions
	pbs.Time.prototype.equals = function(obj){
		if(typeof obj !== 'object'){
			return false;
		}
		if(! obj instanceof pbs.Time){
			return false;
		}
		return obj._hours === this._hours && obj._minutes === this._minutes && obj._seconds === this._seconds;
	};
	pbs.Time.prototype.compareTo = function(obj){
		// make sure we have a valid object to test
		if(!(typeof obj === 'object' && obj instanceof pbs.Time)){
			return NaN;
		}
		
		// check if the hours are different
		if(this._hours < obj._hours){
			return -1;
		}
		if(this._hours > obj._hours){
			return 1;
		}
		
		// if we got here, the hours are the same, so check the minutes
		if(this._minutes < obj._minutes){
			return -1;
		}
		if(this._minutes > obj._minutes){
			return 1;
		}
		
		// if we got here, the hours and minutes are the same, so check the seconds
		if(this._seconds < obj._seconds){
			return -1;
		}
		if(this._seconds > obj._seconds){
			return 1;
		}
		
		// if we got here the two times are equal, so return 0
		return 0;
	};
	pbs.Time.prototype.isBefore = function(obj){
		return this.compareTo(obj) === -1;
	}
	pbs.Time.prototype.isAfter = function(obj){
		return this.compareTo(obj) === 1;
	}
	
	//
	// === Define the Date Prototype ===
	//
	
	// the constructor
	pbs.Date = function(d, m, y){
		// init data with default values
		this._day = 1;
		this._month = 1;
		this._year = 1970;
		
		// deal with any passed args
		if(typeof y !== 'undefined'){
			this.year(y);
		}
		if(typeof m !== 'undefined'){
			this.month(m);
		}
		if(typeof d !== 'undefined'){
			this.day(d);
		}
	};
	
	// the accessor methods
	pbs.Date.prototype.day = function(d){
		if(arguments.length === 0){
			return this._day;
		}
		if(!isValidInteger(d, 1, 31)){
			throw new TypeError('the day must be an integer between 1 and 31 inclusive');
		}
		d = parseInt(d); // force to number if string of digits
		if(!isValidateDMYCombo(d, this._month, this._year)){
			throw new Error('invalid day, month, year combination');
		}
		this._day = d;
		return this;
	};
	pbs.Date.prototype.month = function(m){
		if(arguments.length === 0){
			return this._month;
		}
		if(!isValidInteger(m, 1, 12)){
			throw new TypeError('the month must be an integer between 1 and 12 inclusive');
		}
		m = parseInt(m); // force to number if string of digits
		if(!isValidateDMYCombo(this._day, m, this._year)){
			throw new Error('invalid day, month, year combination');
		}
		this._month = m;
		return this;
	};
	pbs.Date.prototype.year = function(y){
		if(arguments.length === 0){
			return this._year;
		}
		if(!isValidInteger(y)){ // no bounds check on the year
			throw new TypeError('the year must be an integer');
		}
		y = parseInt(y); // force to number if string of digits
		if(!isValidateDMYCombo(this._day, this._month, y)){
			throw new Error('invalid day, month, year combination');
		}
		this._year = y;
		return this;
	};
	
	// define needed functions
	pbs.Date.prototype.international = function(y, m, d){
		if(arguments.length === 0){
			// we are in 'get' mode
			return intToPaddedString(this._year, 4) + '-' + intToPaddedString(this._month, 2) + '-' + intToPaddedString(this._day, 2);
		}
		
		// if we got here we are in 'set' mode
			
		// validate the three pieces of data
		if(!(isValidInteger(d, 1, 31) && isValidInteger(m, 1, 12) && isValidInteger(y))){
			throw new TypeError('invalid date information - must be three integers');
		}
			
		// force the three pieces of data to be numbers and not strings
		d = parseInt(d);
		m = parseInt(m);
		y = parseInt(y);
			
		// test the combination is valid
		if(!isValidateDMYCombo(d, m, y)){
			throw new Error('invalid day, month, year combination');
		}
			
		// set the three pieces of data
		this._day = d;
		this._month = m;
		this._year = y;
		
		// return a refernce to self
		return this;
	};
	pbs.Date.prototype.american = function(m, d, y){
		if(arguments.length === 0){
			// we are in 'get' mode
			var ans = '';
			ans += this._month + '/' + this._day + '/';
			if(this._year <= 0){
				ans += (Math.abs(this._year - 1)) + 'BC';
			}else{
				ans += this._year;
			}
			return ans;
		}
		
		// if we got here we are in 'set' mode
		return this.international(y, m, d); // avoid needless duplication
	};
	pbs.Date.prototype.european = function(d, m, y){
		if(arguments.length === 0){
			// we are in 'get' mode
			var ans = '';
			ans += intToPaddedString(this._day, 2) + '-' + intToPaddedString(this._month, 2) + '-';
			if(this._year <= 0){
				ans += Math.abs(this._year - 1) + 'BCE';
			}else{
				ans += this._year;
			}
			return ans;
		}
		
		// if we got here we are in 'set' mode
		return this.international(y, m, d); // avoid needless duplication
	};
	pbs.Date.prototype.english = function(){
		var ans = '';
		ans += this._day + toOrdinalString(this._day) + ' of ' + monthNameLookup[this._month] + ' ';
		if(this._year <= 0){
			ans += Math.abs(this._year - 1) + 'BCE';
		}else{
			ans += this._year;
		}
		return ans;
	};
	
	// provide a toString
	pbs.Date.prototype.toString = pbs.Date.prototype.international;
	
	// define a clone function
	pbs.Date.prototype.clone = function(){
		return new pbs.Date(this._day, this._month, this._year);
	};
	
	// define comparison functions
	pbs.Date.prototype.equals = function(obj){
		if(typeof obj !== 'object'){
			return false;
		}
		if(! obj instanceof pbs.Date){
			return false;
		}
		return obj._day === this._day && obj._month === this._month && obj._year === this._year;
	};
	pbs.Date.prototype.compareTo = function(obj){
		// make sure we have a valid object to test
		if(!(typeof obj === 'object' && obj instanceof pbs.Date)){
			return NaN;
		}
		
		// check if the years are different
		if(this._year < obj._year){
			return -1;
		}
		if(this._year > obj._year){
			return 1;
		}
		
		// if we got here, the years are the same, so check the months
		if(this._month < obj._month){
			return -1;
		}
		if(this._month > obj._month){
			return 1;
		}
		
		// if we got here, the years and months are the same, so check the days
		if(this._day < obj._day){
			return -1;
		}
		if(this._day > obj._day){
			return 1;
		}
		
		// if we got here the two dates are equal, so return 0
		return 0;
	};
	pbs.Date.prototype.isBefore = function(obj){
		return this.compareTo(obj) === -1;
	};
	pbs.Date.prototype.isAfter = function(obj){
		return this.compareTo(obj) === 1;
	};
	
	// add the static function isLeapYear
	pbs.Date.isLeapYear = function(y){
		// make sure we were passed a plausible year
		if(!isValidInteger(y)){
			throw new TypeError('the year must be an integer');
		}
		
		// figure out if the year is a leapyear or not
		if(y % 4 === 0){
			// year is divisible by 4, so might be a leap year
			if(y % 100 === 0){
				// a century, so not a leap year unless divisible by 400
				if(y % 400 === 0){
					return true;
				}
			}else{
				// divisible by four and not a century, so a leap year
				return true;
			}
		}
		
		// if we got here, the year is not a leap year
		return false;
	};
	
	//
	// === Define the DateTime Prototype ===
	//
	
	// the constructor
	pbs.DateTime = function(d, t){
		// init data with defaults
		this._date = new pbs.Date();
		this._time = new pbs.Time();
		
		// deal with any args that were passed
		if(typeof d !== 'undefined'){
			this.date(d);
		}
		if(typeof t !== 'undefined'){
			this.time(t);
		}
	};
		
	// accessor methods
	pbs.DateTime.prototype.date = function(d){
		if(arguments.length === 0){
			return this._date.clone();
		}
		if(!(d instanceof pbs.Date)){
			throw new TypeError('require an instance of the pbs.Date prototype');
		}
		this._date = d.clone();
		return this;
	};
	pbs.DateTime.prototype.time = function(t){
		if(arguments.length === 0){
			return this._time.clone();
		}
		if(!(t instanceof pbs.Time)){
			throw new TypeError('require an instance of the pbs.Time prototype');
		}
		this._time = t.clone();
		return this;
	};
	
	// define functions
	pbs.DateTime.prototype.american12Hour = function(){
		return this._date.american() + ' ' + this._time.time12();
	};
	pbs.DateTime.prototype.american24Hour = function(){
		return this._date.american() + ' ' + this._time.time24();
	};
	pbs.DateTime.prototype.european12Hour = function(){
		return this._date.european() + ' ' + this._time.time12();
	};
	pbs.DateTime.prototype.european24Hour = function(){
		return this._date.european() + ' ' + this._time.time24();
	};
	
	// provide a toString
	pbs.DateTime.prototype.toString = function(){
		return this._date.toString() + ' ' + this._time.toString();
	};
	
	// define a clone function
	pbs.DateTime.prototype.clone = function(){
		return new pbs.DateTime(this._date, this._time);
	};
	
	// define comparison functions
	pbs.DateTime.prototype.equals = function(obj){
		if(typeof obj !== 'object'){
			return false;
		}
		if(! obj instanceof pbs.DateTime){
			return false;
		}
		return this._date.equals(obj._date) && this._time.equals(obj._time);
	};
	pbs.DateTime.prototype.compareTo = function(obj){
		// make sure we have a valid object to test
		if(!(typeof obj === 'object' && obj instanceof pbs.DateTime)){
			return NaN;
		}
		
		// check if the dates are different
		var dateCompare = this._date.compareTo(obj._date);
		if(dateCompare !== 0){
			return dateCompare;
		}
		
		// if we got here, the dates are the same, so check the times
		var timeCompare = this._time.compareTo(obj._time);
		if(timeCompare !== 0){
			return timeCompare;
		}
		
		// if we got here the two date times are equal, so return 0
		return 0;
	};
	pbs.DateTime.prototype.isBefore = function(obj){
		return this.compareTo(obj) === -1;
	}
	pbs.DateTime.prototype.isAfter = function(obj){
		return this.compareTo(obj) === 1;
	}
})(pbs);